﻿<#
.SYNOPSIS 
Get changed (non-inherited) ACLs (access control lists) on the selected folder and its subfolders. It will also export the results.

.DESCRIPTION
This Powershell script get all changed ACLs on a folder and its subfolders. It will also export its results.
It will generate an output file with a summary of each folder or a fully detailed file with all the non-inherited access rights.

.PARAMETER Path
This option is mandatory. Specifies the path check for changed ACLs.

.PARAMETER CSVDelimiter
Use one of the following parameters:
","  - comma
";"  - semicolon: this is the default option
" "  - space
"`t" - tab
"|"  - pipe
":"  - colon
This will be used as the delimiter in the CSV file.

.PARAMETER LogFile
This specifies the name of the log file.
The default option will save the file in the folder the script is ran from.
It will use the name "GetChangedAccessRights.txt".
Summary or full details will be logged for each folder without inheritance.
If the verbose switch is used, it will log each folder being processed.

.PARAMETER ExportFile
This specifies the name of the export file.
The default option will save the file in the folder the script is ran from.
It will use the name "ChangedACLs.csv" (unless the FullDetails switch is used).

.PARAMETER FullDetails
Set this switch to include the access rights as they are set.
If this switch is used, the text "-FullDetails" will be added to the file
(in case of the default settings: "ChangedACLs-FullDetails.csv")

.INPUTS
None. You cannot pipe objects.

.OUTPUTS
None. No object outputs.

.EXAMPLE
C:\PS> .\GetChangedACLs.ps1 -Path "D:\PowershellAdministrator"
This will recursively get the folder contents from "the D:\PowershellAdministrator" folder.
It will use the default CSV delimiter (a semicolon).
The log file will be saved in the script folder with the name "GetChangedAccessRights.txt"
This will give a summary export of all folders with non-inherited ACLs, which will be saved in the script folder as "ChangedACLs.csv".

.EXAMPLE
C:\PS> .\GetChangedACLs.ps1 -Path "D:\PowershellAdministrator" -CSVDelimiter "," -LogFile "D:\PowershellAdministrator\Logs\GetChangedACLsLog.txt"
This will recursively get the folder contents from "the D:\PowershellAdministrator" folder.
It will use a comma as CSV delimiter.
The log file will be saved in "D:\PowershellAdministrator\Logs\GetChangedACLsLog.txt"
This will give a summary export of all folders with non-inherited ACLs, which will be saved in the script folder as "ChangedACLs.csv".

.EXAMPLE
C:\PS> .\GetChangedACLs.ps1 -Path "D:\PowershellAdministrator" -CSVDelimiter "`t" -ExportFile "D:\PowershellAdministrator\myexport.csv" -FullDetails
This will recursively get the folder contents from "the D:\PowershellAdministrator" folder.
It will use a tab as CSV delimiter.
The log file will be saved in the script folder with the name "GetChangedAccessRights.txt"
This will give a fully detailed export of all folders with non-inherited ACLs, which will be saved as "D:\PowershellAdministrator\myexport-FullDetails.csv".

.LINK
https://powershelladministrator.com/2020/03/21/get-acl-on-non-inherited-folders/
https://www.PowershellAdministrator.com
#>

Param(
[Parameter(Mandatory=$true)][string]$Path,
#only these delimiters can be set: comma, semicolon, space, tab, pipe, colon
[ValidateSet(",",";"," ","`t","|",":")][char]$CSVDelimiter = ";",
[string]$LogFile="$PSScriptRoot\GetChangedAccessRights.txt",
[string]$ExportFile="$PSScriptRoot\ChangedACLs.csv",
[Switch]$FullDetails
)
#If the script is ran in Powershell ISE, the $PSScriptRoot variable is not populated at the param level, thus in that case it needs to be set when the script runs
If($LogFile.StartsWith("\")) { $LogFile = "$PSScriptRoot$LogFile" } 
If($ExportFile.StartsWith("\")) { $ExportFile = "$PSScriptRoot$ExportFile" } 

#Start writing to the log file
Start-Transcript -Path $LogFile

#Get the list of folders
$Dirs = Get-ChildItem "$Path" -directory -recurse -ErrorAction SilentlyContinue
#Create empty array
$DirectoryListWithChangedACL = @()

#If the switch "FullDetails" is present, add -FullDetails to the file name
If($FullDetails.IsPresent) { $ExportFile = $ExportFile.Replace(".csv","-FullDetails.csv") }

#Show the name of the export file
Write-Host ("Export file = {0}" -f $ExportFile)

#Loop through the folders
Foreach ($dir in $Dirs)
{
    #In case of verbose logging, show each folder being processed
    Write-Verbose ("processing folder {0}" -f $dir.FullName)
    #Get the ACL information, if there is no inheritance
    $aclCheck = Get-Acl -Path $dir.FullName | where {$_.Access.IsInherited -eq $false} -ErrorAction SilentlyContinue
    If($aclCheck)
    {
        #Non-inherited ACL found. Output this to the screen for the log file
        Write-Host ("Found a changed ACL on folder {0}" -f $dir.FullName)
        #Get the details if the FullDetails switch is set
        If($FullDetails.IsPresent)
        {
            #In case of verbose logging, Show the ACL information
            Write-Verbose ($aclCheck.Access | Select-Object IdentityReference,AccessControlType,FileSystemRights | ft -AutoSize | Out-String)
            Foreach($user in $aclCheck.Access)
            {
                #Create a new psobject and add the properties to the psobject
                $DirectoryList = New-Object psobject
                Add-Member -InputObject $DirectoryList -MemberType NoteProperty -Name "FolderWithChangedACL" -Value $dir.FullName
                Add-Member -InputObject $DirectoryList -MemberType NoteProperty -Name "Identity" -Value $user.IdentityReference
                Add-Member -InputObject $DirectoryList -MemberType NoteProperty -Name "AccessControlType" -Value $user.AccessControlType
                Add-Member -InputObject $DirectoryList -MemberType NoteProperty -Name "FileSystemRights" -Value $user.FileSystemRights
                #Add the psobject to the array of objects
                $DirectoryListWithChangedACL += $DirectoryList
            }
        }
        Else
        {
            #Add the properties to the psobjec
            $DirectoryList = New-Object psobject
            Add-Member -InputObject $DirectoryList -MemberType noteproperty -Name "FolderWithChangedACL" -Value $dir.FullName
            #Add the psobject to the array of objects
            $DirectoryListWithChangedACL += $DirectoryList
        }
    }
}
#Export the array of objects to a CSV file
$DirectoryListWithChangedACL | Export-Csv $ExportFile -Delimiter $CSVDelimiter -NoTypeInformation
#Stop writing to the log file
Stop-Transcript