# **********************************************************************************************************
# ***
# *** Scriptname: Create-ExcelFile.ps1
# ***
# *** Author: OcinO - Powershell Administrator
# ***
# *** Version: 1.2
# ***
# *** History: 21-11-15 Added autofilter and lock top rows
# ***          15-10-14 Initial version (1.0) 
# ***
# *** 2015 Powershell Administrator. Use at your own risk.
# ***
# **********************************************************************************************************

$IsAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")
$CanBeAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]"$env:USERNAME").IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")

#Start the script as Administrator, because of -IncludeUserName switch for get-process (and getting CPU info on all processes)
If(!($IsAdmin))
{
    If($CanBeAdmin)
    {
        $Arguments = "& '" + $MyInvocation.MyCommand.Definition + "'"
        Start-Process powershell -Verb runAs -ArgumentList $Arguments
    }
    Else
    {
        Write-Error "You have to be an administrator to run this script" -ErrorAction Stop
    }
    Break
}

#Set the current thread to en-US, otherwise the Excel com object will give many errors
[System.Threading.Thread]::CurrentThread.CurrentCulture = "en-US"
$ExcelFile = "D:\PowershellAdministrator\Processes.xlsx"
#Remove the file if it exists
If(Test-Path $ExcelFile) { Remove-Item -Path $ExcelFile -Force }

#New Excel Application
$Excel = New-Object -ComObject Excel.Application
#Do no show Excel; set to true to show the Excel window typing the information, but user input can interfere with the script
$Excel.visible = $false

If(Test-Path $ExcelFile)
{
    #Open the document
    $WorkBook = $Excel.WorkBooks.Open($ExcelFile)
    #Set the sheet name
    $Sheet1 = $Excel.Worksheets.Item(1)
    $Sheet1.Name = "ProcessInformation"
}
Else
{ 
    #Create new workbook (which contains 1 sheet by default, from Excel 2007 and on)
    $WorkBook = $Excel.Workbooks.Add()
    #Set the sheet name
    $Sheet1 = $Excel.Worksheets.Item(1)
    $Sheet1.Name = "ProcessInformation"
}

#Create no extra worksheets; uncommment to create an extra one
#$Sheet2 = $Excel.Worksheets.Add()
#$Sheet2 = $Excel.Worksheets.Item(2)
#$Sheet2.Name = "Sheet2"

#Create Heading for process information
$Sheet1.Cells.Item(1,1) = "Id"
$Sheet1.Cells.Item(1,2) = "ProcessName"
$Sheet1.Cells.Item(1,3) = "SessionId"
$Sheet1.Cells.Item(1,4) = "UserName"
$Sheet1.Cells.Item(1,5) = "Cpu"

#Create the missing type
$MissingType = [System.Type]::Missing

#Create a collection of sheets, separate by comma.
$colSheets = ($Sheet1)
#Set the first row to have bold text and set auto filter. Autofit column in the end
foreach ($sheet in $colSheets)
{
    #Lock the first row
    $sheet.Application.ActiveWindow.SplitColumn = 0
    $sheet.Application.ActiveWindow.SplitRow = 1
    $sheet.Application.ActiveWindow.FreezePanes = $true
    
    #Set other sheet top row defaults
    $Range = $sheet.UsedRange
    #Uncomment to set back color
    #$Range.Interior.ColorIndex = 20
    #Set the font to bold and color to blue
    $Range.Font.ColorIndex = 11
    $Range.Font.Bold = $True
    #Set autofilter
    #Field, Criteria1, XlAutoFilterOperator Operator, Criteria2, VisibleDropDown
    #XLAutoFilterOperators: xlAnd *default, xlBottom10Items, xlBottom10Percent, xlOr, xlTop10Items, xlTop10Percent
    $Range.AutoFilter(1, $MissingType, 1, $MissingType, $MissingType) | Out-Null
    $Range.EntireColumn.AutoFit() | Out-Null
}

#Get a list of processes with user name (need to be admin to use this switch)
$Processes = Get-Process -IncludeUserName

#Since the first row is for the titles, we start with row 2
$Row = 2

Foreach($Process in $Processes)
{
    #Don't want my account user name ending up in the example file; thus I replace it
    If($Process.UserName -eq "$env:USERDOMAIN\$env:USERNAME") { $Process.UserName = "PowershellAdministrator" }
    
    #Write the process information to the sheet
    $Sheet1.Cells.Item($Row, 1) = $Process.Id
    $Sheet1.Cells.Item($Row, 2) = $Process.Name
    $Sheet1.Cells.Item($Row, 3) = $Process.SessionId
    $Sheet1.Cells.Item($Row, 4) = $Process.UserName
    $Sheet1.Cells.Item($Row, 5) = $Process.CPU
    $Row++
}

#Auto fit the cells
foreach ($sheet in $colSheets)
{
    $Range = $sheet.UsedRange															
    $Range.EntireColumn.AutoFit() | Out-Null
}
#Save the workbook
If(Test-Path $ExcelFile) { $WorkBook.Save() }
Else { $WorkBook.SaveAs($ExcelFile) } 
#Close Excel and release objects
$Excel.Quit()
[System.Runtime.Interopservices.Marshal]::ReleaseComObject($Excel) | out-null
Remove-Variable -Name Excel -Scope Global -EA 0
